/**

VIDEO ANALYSIS PREPROCESSING  - BASE CLASSES
author: Pavel Zak, izakpa@fit.vutbr.cz

The preprocessing reads the registered queues from the shared memory (SMInterface) and provides the SM with desired output images.

The preprocessing can be extended with new image operations by adding new preprocessing modules - see preprocModules.h

*/

//TODOs
// optimize module tree
// module parameters check

#ifndef __preprocessing_h__
#define __preprocessing_h__

#include <string>
#include <vector>
#include <iostream>
#include <map>

#ifdef USE_IPP
	#include <ippcc.h>
#endif


#include "boost/tokenizer.hpp"

#include "SMInterface.h"

#include "cv.h"
#include "highgui.h"


#include "preprocModules.h"

#define LINUX
#ifdef LINUX

#ifndef UINT8
#  include <stdint.h>
  typedef uint8_t UINT8;
#endif

#endif
//source_!_operation1_param1_val1_param2_val2_!_operation2_param1_val1 ...

using namespace std;

/**
* Transforms YUV444 pixel values to RGB format
*/
void YUV4442RGB(UINT8 Y, UINT8 U, UINT8 V, UINT8 &R, UINT8 &G, UINT8 &B);

/**
* Transforms YUV420 pixel values to RGB format. Pixel position is referenced by coords x, y.
* @param p_data pointer to the beginning of image data
*/
void YUV4202RGB(UINT8 * p_data, int x, int y, int width, int height, UINT8 &R, UINT8 &G, UINT8 &B);

/**
* Creates OpenCV RGB IplImage from the YUV420 data buffer
* @param p_data Pointer to the image data origin
* @param image returned IplImage structure filled with data
* @param width width of the image
* @param height height of the image
* @return 0
*/
int CreateIplFromYUVbuffer(UINT8 * p_data, IplImage ** image, int width, int height);
int CreateGrayIplFromYUVbuffer(UINT8 * p_data, IplImage ** image, int width, int height);


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

/**
* Helper class that contains image sources with information
*/
class Preproc_image{
protected:
	IplImage * image;

	string sourceID;
	int64 frameNo;
	//int inited();
public:
	Preproc_image();
	Preproc_image(string source);

	//copy construktor
	Preproc_image(const Preproc_image &preproc);

	//assignment
	Preproc_image & operator= (const Preproc_image & preproc);

	~Preproc_image();

	//int load();
	void  update(IplImage * act_image, int64 timestamp);
	IplImage * getImage();
	
	string getSource();
	int64 getTimestamp();
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

/**
* Main preprocessing class
* Constructs and maintains the preprocessing tree, creation as well as a communication with shared memory.
*
* typical use 
* - when new source image is available - addSourceFrame(...) is called
* - than we call processSM() to process the images and uploads them to SharedMemory
*/
class Preprocessing{
protected:
	TSMObject * sm;
	vector<Preproc_module *> proc_trees;
	vector<Preproc_module *> terminals;
	vector< string > queues;

	map<string, Preproc_image> image_sources;
	
	//IplImage * act_frame;
	//int64 timestamp;
	Preproc_ModuleFactory factory;

	

	int processImages();
	int uploadImages();

	int buildTree(map< string, bool> queues);
	int buildProcQueue(string qid);

	int processImage(IplImage * image, int64 frameNo, string sourceID);

public:


	Preprocessing();
	~Preprocessing();

/**	
* You can register some default image queues
*/
	void register_queue(string queue);

/**
* Asks the SM for active queues and uploads back the desired results.
* In fact - EVERYTHING is done here.
*/
	int processSM();

/**
* Adds the source frame that could be processed.
* @param image source image
* @param frameNo timestamp of current image
* @param sourceID source identifier
*/

	int addSourceFrame(IplImage * image, string sourceID, int64 timestamp);

	int addSourceFrameYUV(UINT8 * p_data, int x, int y, string sourceID, int64 timestamp);

/**
* DEBUG ONLY !!
* displays all the results of the processing tree
*/
void showResults();

};

#endif
